from flask import Flask, request, jsonify
from PIL import Image, ImageDraw, ImageFont
import io
import base64
import json
import logging
import os
import google.generativeai as genai
from typing import List, Dict
from pathlib import Path

app = Flask(__name__)

# 設定日誌
logging.basicConfig(
    level=logging.DEBUG,
    format='%(asctime)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

# 語言對照表
LANGUAGE_MAP = {
    "En": {"name": "English", "native": "English"},
    "Fr": {"name": "French", "native": "Français"},
    "Pt": {"name": "Portuguese", "native": "Português"},
    "De": {"name": "German", "native": "Deutsch"},
    "El": {"name": "Greek", "native": "Ελληνικά"},
    "Es": {"name": "Spanish", "native": "Español"},
    "Cs": {"name": "Czech", "native": "Čeština"},
    "Da": {"name": "Danish", "native": "Dansk"},
    "Hr": {"name": "Croatian", "native": "Hrvatski"},
    "Hu": {"name": "Hungarian", "native": "Magyar"},
    "It": {"name": "Italian", "native": "Italiano"},
    "Ja": {"name": "Japanese", "native": "日本語"},
    "Ko": {"name": "Korean", "native": "한국어"},
    "Nl": {"name": "Dutch", "native": "Nederlands"},
    "Nn": {"name": "Norwegian", "native": "Norsk"},
    "Po": {"name": "Polish", "native": "Polski"},
    "Ro": {"name": "Romanian", "native": "Română"},
    "Ru": {"name": "Russian", "native": "Русский"},
    "Sv": {"name": "Swedish", "native": "Svenska"},
    "Tr": {"name": "Turkish", "native": "Türkçe"},
    "Zh": {"name": "Traditional Chinese", "native": "繁體中文"},
}

# 預設值
DEFAULT_MODEL_NAME = "gemini-2.0-flash"
DEFAULT_OUTPUT_WIDTH = 1920
DEFAULT_OUTPUT_HEIGHT = 1080
DEFAULT_FONT_SIZE = 48

# 字體設定
FONT_PATHS = {
    "Zh": "NotoSansTC-Bold.ttf",  # 繁體中文
    "Ja": "NotoSansJP-Bold.ttf",  # 日文
    "Ko": "NotoSansKR-Bold.ttf",  # 韓文
    "default": "NotoSansTC-Bold.ttf"  # 預設字體
}

# 圖片儲存設定
SAVE_DIR = "received_images"  # 儲存目錄

# 全域變數(從 .env 讀取)
MODEL_NAME = None
OUTPUT_WIDTH = None
OUTPUT_HEIGHT = None
FONT_SIZE = None
SAVE_IMAGES = False
GEMINI_API_KEY = None
fonts = {}  # 儲存不同語言的字體
model = None


def load_env():
    """載入 .env 檔案中的環境變數"""
    global GEMINI_API_KEY, SAVE_IMAGES, MODEL_NAME, OUTPUT_WIDTH, OUTPUT_HEIGHT, FONT_SIZE
    
    # 設定預設值
    MODEL_NAME = DEFAULT_MODEL_NAME
    OUTPUT_WIDTH = DEFAULT_OUTPUT_WIDTH
    OUTPUT_HEIGHT = DEFAULT_OUTPUT_HEIGHT
    FONT_SIZE = DEFAULT_FONT_SIZE
    
    env_file = Path('.env')
    
    # 優先從 .env 檔案讀取
    if env_file.exists():
        logger.info("Loading environment variables from .env file")
        with open(env_file, 'r', encoding='utf-8') as f:
            for line in f:
                line = line.strip()
                if line and not line.startswith('#') and '=' in line:
                    key, value = line.split('=', 1)
                    key = key.strip()
                    value = value.strip().strip('"').strip("'")
                    
                    if key == 'GEMINI_API_KEY':
                        GEMINI_API_KEY = value
                        logger.info("GEMINI_API_KEY loaded from .env file")
                    elif key == 'SAVE_IMAGES':
                        SAVE_IMAGES = value.lower() in ('true', '1', 'yes')
                        logger.info(f"SAVE_IMAGES set to {SAVE_IMAGES}")
                    elif key == 'MODEL_NAME':
                        MODEL_NAME = value
                        logger.info(f"MODEL_NAME set to {MODEL_NAME}")
                    elif key == 'OUTPUT_WIDTH':
                        try:
                            OUTPUT_WIDTH = int(value)
                            logger.info(f"OUTPUT_WIDTH set to {OUTPUT_WIDTH}")
                        except ValueError:
                            logger.warning(f"Invalid OUTPUT_WIDTH value: {value}, using default {DEFAULT_OUTPUT_WIDTH}")
                    elif key == 'OUTPUT_HEIGHT':
                        try:
                            OUTPUT_HEIGHT = int(value)
                            logger.info(f"OUTPUT_HEIGHT set to {OUTPUT_HEIGHT}")
                        except ValueError:
                            logger.warning(f"Invalid OUTPUT_HEIGHT value: {value}, using default {DEFAULT_OUTPUT_HEIGHT}")
                    elif key == 'FONT_SIZE':
                        try:
                            FONT_SIZE = int(value)
                            logger.info(f"FONT_SIZE set to {FONT_SIZE}")
                        except ValueError:
                            logger.warning(f"Invalid FONT_SIZE value: {value}, using default {DEFAULT_FONT_SIZE}")
    
    # 如果 .env 中沒有 GEMINI_API_KEY,則從環境變數讀取
    if not GEMINI_API_KEY:
        GEMINI_API_KEY = os.getenv("GEMINI_API_KEY")
        if GEMINI_API_KEY:
            logger.info("GEMINI_API_KEY loaded from environment variable")
    
    if not GEMINI_API_KEY:
        raise ValueError("GEMINI_API_KEY not found in .env file or environment variables")
    
    # 顯示最終設定
    logger.info(f"Final configuration: MODEL={MODEL_NAME}, SIZE={OUTPUT_WIDTH}x{OUTPUT_HEIGHT}, FONT={FONT_SIZE}")


def load_font():
    """載入字體 - 載入中文、日文、韓文字體"""
    global fonts
    
    for lang, font_path in FONT_PATHS.items():
        try:
            fonts[lang] = ImageFont.truetype(font_path, FONT_SIZE)
            logger.info(f"Font loaded successfully for {lang}: {font_path}")
        except Exception as e:
            logger.warning(f"Failed to load font for {lang} from {font_path}: {e}")
            # 如果載入失敗，使用預設字體
            if lang != "default" and "default" in fonts:
                fonts[lang] = fonts["default"]
                logger.info(f"Using default font for {lang}")


def get_font_for_language(target_lang: str):
    """根據目標語言取得對應字體"""
    if target_lang in fonts:
        return fonts[target_lang]
    elif "default" in fonts:
        logger.debug(f"No specific font for {target_lang}, using default")
        return fonts["default"]
    else:
        logger.warning("No fonts loaded, using PIL default font")
        return ImageFont.load_default()


def init_gemini():
    """初始化 Gemini API"""
    global model
    
    genai.configure(api_key=GEMINI_API_KEY)
    model = genai.GenerativeModel(MODEL_NAME)
    logger.info(f"Gemini model initialized: {MODEL_NAME}")


def create_save_directory():
    """建立圖片儲存目錄"""
    if SAVE_IMAGES:
        save_path = Path(SAVE_DIR)
        save_path.mkdir(exist_ok=True)
        logger.info(f"Image save directory created/verified: {SAVE_DIR}")


def save_received_image(image_base64: str, target_lang: str) -> tuple:
    """儲存接收到的圖片
    
    Returns:
        tuple: (received_filepath, base_filename) 或 (None, None)
    """
    if not SAVE_IMAGES:
        return None, None
    
    try:
        from datetime import datetime
        
        # 產生檔名:timestamp_language_received.png
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S_%f")[:-3]
        base_filename = f"{timestamp}_{target_lang}"
        filename = f"{base_filename}_received.png"
        filepath = Path(SAVE_DIR) / filename
        
        # 解碼並儲存圖片
        image_data = base64.b64decode(image_base64)
        with open(filepath, 'wb') as f:
            f.write(image_data)
        
        logger.info(f"Received image saved to {filepath}")
        return str(filepath), base_filename
        
    except Exception as e:
        logger.error(f"Failed to save received image: {e}")
        return None, None


def save_output_image(image_base64: str, base_filename: str) -> str:
    """儲存輸出的覆蓋圖片
    
    Args:
        image_base64: base64 編碼的圖片
        base_filename: 基礎檔名(與 received image 相同)
    
    Returns:
        str: 儲存的檔案路徑或 None
    """
    if not SAVE_IMAGES or not base_filename:
        return None
    
    try:
        filename = f"{base_filename}_output.png"
        filepath = Path(SAVE_DIR) / filename
        
        # 解碼並儲存圖片
        image_data = base64.b64decode(image_base64)
        with open(filepath, 'wb') as f:
            f.write(image_data)
        
        logger.info(f"Output image saved to {filepath}")
        return str(filepath)
        
    except Exception as e:
        logger.error(f"Failed to save output image: {e}")
        return None


def get_prompt_for_language(target_lang: str) -> str:
    """根據目標語言生成對應的 prompt"""
    lang_info = LANGUAGE_MAP.get(target_lang)
    
    if not lang_info:
        # 預設使用繁體中文
        lang_info = LANGUAGE_MAP["Zh"]
        logger.warning(f"Unknown target language '{target_lang}', defaulting to Traditional Chinese")
    
    target_language = lang_info["native"]
    
    # 根據不同語言提供示例
    example_translation = {
        "En": "Hello.",
        "Fr": "Bonjour.",
        "Pt": "Olá.",
        "De": "Hallo.",
        "El": "Γεια σας.",
        "Es": "Hola.",
        "Cs": "Ahoj.",
        "Da": "Hej.",
        "Hr": "Bok.",
        "Hu": "Helló.",
        "It": "Ciao.",
        "Ja": "こんにちは。",
        "Ko": "안녕하세요.",
        "Nl": "Hallo.",
        "Nn": "Hei.",
        "Po": "Cześć.",
        "Ro": "Salut.",
        "Ru": "Привет.",
        "Sv": "Hej.",
        "Tr": "Merhaba.",
        "Zh": "你好。",
    }.get(target_lang, "Hello.")
    
    prompt = f"""Extract the texts in the image and translate into {target_language}. The output should be in JSON format like:
```json
[
    {{
        "location": "message window",
        "original": "こんにちは。",
        "translation": "{example_translation}"
    }}
]
```
"""
    
    return prompt


def call_gemini(image_base64: str, target_lang: str = "Zh") -> str:
    """呼叫 Gemini API 進行圖片分析和翻譯"""
    prompt = get_prompt_for_language(target_lang)
    
    try:
        # 將 base64 字串轉換為圖片
        image_data = base64.b64decode(image_base64)
        image = Image.open(io.BytesIO(image_data))
        
        # 呼叫 Gemini API
        response = model.generate_content([prompt, image])
        
        logger.debug(f"Gemini response: {response.text}")
        return response.text
        
    except Exception as e:
        logger.error(f"Error calling Gemini API: {e}")
        raise


def parse_json_response(response: str) -> List[Dict]:
    """解析 LLM 回應的 JSON"""
    result = response.strip()
    
    # 移除 markdown 程式碼區塊標記
    if result.startswith("```json"):
        result = result[7:]
    elif result.startswith("```"):
        result = result[3:]
    
    if result.endswith("```"):
        result = result[:-3]
    
    result = result.strip()
    
    try:
        return json.loads(result)
    except json.JSONDecodeError as e:
        logger.error(f"Failed to parse JSON: {e}")
        logger.debug(f"Response content: {result}")
        raise


def is_cjk_language(target_lang: str) -> bool:
    """判斷是否為中日韓語系"""
    return target_lang in ['Zh', 'Ja', 'Ko']


def create_overlay_image(translations: List[Dict], target_lang: str = 'Zh') -> str:
    """建立包含翻譯文字的覆蓋圖片
    
    Args:
        translations: 翻譯結果列表
        target_lang: 目標語言代碼
    """
    img = Image.new('RGBA', (OUTPUT_WIDTH, OUTPUT_HEIGHT), (0, 0, 0, 128))
    draw = ImageDraw.Draw(img)
    
    y_pos = 100
    line_height = 48
    
    # 根據語系決定每行字元數
    chars_per_line = 20 if is_cjk_language(target_lang) else 60
    
    # 取得對應語言的字體
    font = get_font_for_language(target_lang)
    
    for translation in translations:
        # 繪製位置 (綠色)
        draw.text((10, y_pos), translation['location'], fill='green', font=font)
        y_pos += line_height
        
        # 繪製翻譯文字 (青色)
        translation_text = translation['translation']
        lines = translation_text.split('\n')
        
        for line in lines:
            # 將長文字分割成每 chars_per_line 字元一行
            chunks = [line[i:i+chars_per_line] for i in range(0, len(line), chars_per_line)]
            for chunk in chunks:
                draw.text((60, y_pos), chunk.strip(), fill='cyan', font=font)
                y_pos += line_height
    
    # 轉換為 base64
    buffer = io.BytesIO()
    img.save(buffer, format='PNG')
    buffer.seek(0)
    
    return base64.b64encode(buffer.getvalue()).decode('utf-8')


@app.route('/', methods=['POST'])
def process_image():
    """處理圖片翻譯請求"""
    try:
        # 取得 target_lang 參數 (從 GET 參數或 JSON body)
        target_lang = request.args.get('target_lang', 'Zh')
        
        # 嘗試解析 JSON
        try:
            data = request.get_json(force=True)
        except Exception as e:
            logger.error(f"Failed to parse JSON: {e}")
            return jsonify({'error': 'Invalid JSON format'}), 400
        
        if not data or 'image' not in data:
            return jsonify({'error': 'No image provided'}), 400
        
        image_base64 = data['image']
        logger.debug(f"Received request with target_lang={target_lang}")
        
        # 儲存接收到的圖片(如果啟用)
        saved_path, base_filename = save_received_image(image_base64, target_lang)
        if saved_path:
            logger.debug(f"Received image saved to: {saved_path}")
        
        # 呼叫 Gemini API 進行翻譯
        gemini_response = call_gemini(image_base64, target_lang)
        logger.debug(f"Result from Gemini: {gemini_response}")
        
        # 解析 JSON 回應
        translations = parse_json_response(gemini_response)
        logger.debug(f"Parsed translations: {translations}")
        
        # 建立覆蓋圖片 (傳入 target_lang 參數)
        output_image = create_overlay_image(translations, target_lang)
        
        # 儲存輸出圖片(如果啟用)
        output_path = save_output_image(output_image, base_filename)
        if output_path:
            logger.debug(f"Output image saved to: {output_path}")
        
        return jsonify({'image': output_image})
        
    except Exception as e:
        logger.error(f"Error processing request: {e}", exc_info=True)
        return jsonify({'error': str(e)}), 500


if __name__ == '__main__':
    # 啟動時初始化
    load_env()
    load_font()
    init_gemini()
    create_save_directory()
    
    logger.info(f"Server starting on port 4404...")
    logger.info(f"Image saving: {'ENABLED' if SAVE_IMAGES else 'DISABLED'}")
    if SAVE_IMAGES:
        logger.info(f"Images will be saved to: {SAVE_DIR}")
    app.run(host='0.0.0.0', port=4404, debug=False)